<?php

namespace App\Http\Controllers;

use App\Models\inventario;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreinventarioRequest;
use App\Http\Requests\UpdateinventarioRequest;
use App\Imports\InventariosImport;
use App\Models\categorias;
use App\Models\equipos;
use App\Models\Gcategorias;
use App\Models\Ginventarios;
use App\Models\Gsedes;
use App\Models\Gsubcategorias;
use App\Models\sede;
use App\Models\servicios;
use App\Models\subcategorias;
use App\Models\terceros;
use App\Models\ubicaciones;
use App\Models\zonas;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Maatwebsite\Excel\Facades\Excel;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

use function Pest\Laravel\json;

class InventarioController extends Controller
{
    /**
     * Summary of index
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(): JsonResponse
    {
        $inventarios = Inventario::all();
        foreach ($inventarios as $inventario) {
            $inventario->cronograma;
            $inventario->archivos;
            $inventario->subcategorias->categorias->sedes;
        }
        return response()->json($inventarios, 200);
    }
    public function obtenerEstadisticas(): JsonResponse
    {
        // Obtener totales
        $totales = [
            'medico' => inventario::count(),
            'general' => Ginventarios::count()
        ];

        // Estadísticas por sede médica
        $porSedeMedico = sede::with(['categorias.subcategorias' => function($query) {
            $query->withCount('inventarios');
        }])
        ->get()
        ->map(function ($sede) {
            return [
                'name' => $sede->nombre,
                'value' => $sede->categorias->flatMap->subcategorias->sum('inventarios_count')
            ];
        });

        // Estadísticas por sede general
        $porSedeGeneral = Gsedes::with(['gcategorias.gsubcategorias' => function($query) {
            $query->withCount('ginventarios');
        }])
        ->get()
        ->map(function ($sede) {
            return [
                'name' => $sede->nombre,
                'value' => $sede->gcategorias->flatMap->gsubcategorias->sum('ginventarios_count')
            ];
        });

        // Categorías médicas
        $categoriasMedico = categorias::with(['subcategorias' => function($query) {
            $query->withCount('inventarios');
        }])
        ->get()
        ->map(function ($categoria) {
            return [
                'name' => $categoria->nombre,
                'value' => $categoria->subcategorias->sum('inventarios_count')
            ];
        });

        // Categorías generales
        $categoriasGeneral = Gcategorias::with(['gsubcategorias' => function($query) {
            $query->withCount('ginventarios');
        }])
        ->get()
        ->map(function ($categoria) {
            return [
                'name' => $categoria->nombre,
                'value' => $categoria->gsubcategorias->sum('ginventarios_count')
            ];
        });

        // Subcategorías médicas
        $subcategoriasMedico = subcategorias::withCount('inventarios')
            ->get()
            ->map(function ($subcategoria) {
                return [
                    'name' => $subcategoria->nombre ?? 'Sin nombre',
                    'value' => $subcategoria->inventarios_count
                ];
            })->filter(function ($item) {
                return !empty($item['name']);
            });

        // Subcategorías generales
        $subcategoriasGeneral = Gsubcategorias::withCount('ginventarios')
            ->get()
            ->map(function ($subcategoria) {
                return [
                    'name' => $subcategoria->nombre ?? 'Sin nombre',
                    'value' => $subcategoria->ginventarios_count
                ];
            })->filter(function ($item) {
                return !empty($item['name']);
            });

        return response()->json([
            'totales' => $totales,
            'porSedeMedico' => $porSedeMedico,
            'porSedeGeneral' => $porSedeGeneral,
            'categoriasMedico' => $categoriasMedico,
            'categoriasGeneral' => $categoriasGeneral,
            'subcategoriasMedico' => $subcategoriasMedico,
            'subcategoriasGeneral' => $subcategoriasGeneral
        ], 200);
    }

    public function getCat(Request $request): JsonResponse
    {
        $perPage = $request->input('per_page', 10);
        if ($request->categoria === '*') {
            $inventarios = Inventario::paginate($perPage);
        } else {
            $inventarios = Inventario::where('subcategoria_id', '=', $request->categoria)
            ->paginate($perPage);
        }
        $inventarios->getCollection()->transform(function ($inventario) {
            $inventario->archivos;
            $inventario->cronograma;
            $inventario->subcategorias->categorias->sedes;
            return $inventario;
        });
        return response()->json($inventarios, 200);
    }

    /**
     * Summary of getData
     * @return \Illuminate\Http\JsonResponse
     */
    public function getData(): JsonResponse
    {
        $response = [
            'equipos' => equipos::all(),
            'sedes' => sede::all(),
            'servicios' => servicios::all(),
            'terceros' => terceros::all(),
            'ubicaciones' => ubicaciones::all(),
            'zonas' => zonas::all(),
            'categorias' => categorias::all(),
            'subcategorias' => subcategorias::all(),
        ];
        return response()->json($response, 200);
    }

    /**
     * Summary of crearQrs
     * @return \Illuminate\Http\JsonResponse
     */
    public function crearQrs(): JsonResponse
    {
        $inventarios = Inventario::whereNull('qr')->get();
        foreach ($inventarios as $inventario) {
            QrCode::format('png')->generate(env('APP_URL') . '/inventario/show?id=' . $inventario->id, '../public/qrcodes/' . $inventario->id . '.png');
            $inventario->qr = env('QR_URL') . $inventario->id . '.png';
            $inventario->save();
        }
        return response()->json($inventarios, 200);
    }

    /**
     * Summary of importData
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function importData(Request $request): JsonResponse
    {
        if ($request->hasFile('excel')) {
            $array = Excel::toArray(new InventariosImport, $request->file('excel'));
            $response = [];
            try {
                foreach ($array[0] as $element) {
                    $inventario = new inventario();
                    $sede = sede::firstWhere('nombre', $element['sede']);
                    if ($sede == NULL) {
                        $sede = new Sede;
                        $sede->nombre = $element['sede'];
                        $sede->save();
                    }
                    $categoria = categorias::firstWhere('nombre', $element['categoria']);
                    if ($categoria == NULL) {
                        $categoria = new categorias();
                        $categoria->nombre = $element['categoria'];
                        $categoria->sede_id = $sede['id'];
                        $categoria->save();
                    }
                    $subcategoria = subcategorias::firstWhere('nombre', $element['subcategoria']);
                    if ($subcategoria == NULL) {
                        $subcategoria = new subcategorias();
                        $subcategoria->nombre = $element['subcategoria'];
                        $subcategoria->categoria_id = $categoria['id'];
                        $subcategoria->save();
                    }
                    $inventario->subcategoria_id = $subcategoria['id'];
                    $inventario->ubicacion = $element['ubicacion'];
                    $inventario->marca = $element['marca'];
                    $inventario->servicio = $element['servicio'];
                    $inventario->nombre = $element['nombre'];
                    $inventario->comercio = $element['permiso'];
                    $inventario->riesgo = $element['riesgo'];
                    $inventario->modelo = $element['modelo'];
                    $inventario->serie = $element['serie'];
                    $inventario->placa = $element['placa'];
                    $inventario->modalidad = $element['modalidad'];
                    $inventario->save();
                    QrCode::format('png')->generate(env('APP_URL') . '/inventario/show?id=' . $inventario->id, '../public/qrcodes/' . $inventario->id . '.png');
                    $inventario->qr = env('QR_URL') . $inventario->id . '.png';
                    $inventario->save();
                    $response[] = $inventario;
                }
                return response()->json($response, 200);
            } catch (ModelNotFoundException $exception) {
                return response()->json(['message' => $exception->getMessage()], 500);
            }
            
        }
        return response()->json(['message' => "No se encontro el archivo"], 404);
    }

    /**
     * Summary of ecportBase
     * @param \Illuminate\Http\Request $request
     * @return mixed|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportBase(Request $request)
    {
        return response()->streamDownload(function () {
    ob_end_clean(); // Elimina cualquier salida previa
    flush(); // Limpia el buffer de salida
    readfile(public_path('cargaArchivosMasivos.xlsx'));
}, 'cargaArchivosMasivos.xlsx', [
    'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
]);

    }

    public function downloadQr(Request $request)
    {
        return response()->download(public_path('qrcodes/'. $request->id . '.png'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreinventarioRequest $request): JsonResponse
    {
        try {
            $inventario = new inventario();
            $inventario->fill($request->all());
            $inventario->saveOrFail();
            QrCode::format('png')->generate(env('APP_URL') . '/inventario/show?id=' . $inventario->id, '../public/qrcodes/' . $inventario->id . '.png');
            $inventario->qr = env('QR_URL') . $inventario->id . '.png';
            $inventario->saveOrFail();
            return response()->json(['message' => 'Inventario agregado exitosamente', 'product' => $inventario], 201);
        } catch (ModelNotFoundException $exception) {
            return response()->json(['message' => $exception->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(inventario $inventario): JsonResponse
    {
        $inventario->cronograma; 
        $inventario->archivos; 
        $inventario->subcategorias->categorias->sedes;
        return response()->json($inventario, 200);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(inventario $inventario)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateinventarioRequest $request, inventario $inventario): JsonResponse
    {
        $inventario->fill($request->all());
        if ($inventario->save()) {
            return response()->json(['data' => $inventario, 'message' => 'El inventario se ha actualizado correctamente'], 200);
        }
        return response()->json(['data' => $inventario, 'message' => 'Ha ocurrido un error al actualizar el inventario'], 500);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(inventario $inventario)
    {
        $inventario->delete();
        return response()->json(['data' => $inventario, 'message' => 'El inventario se ha eliminado correctamente'], 200);
    }
}
